/**
 * Options 页面脚本 - 纯 JavaScript 实现
 */

// ==================== 默认设置 ====================
const sharedConfig = globalThis.ChatTOCConfig || {}
const defaultSettings = sharedConfig.defaultSettings || {}

function normalizeSettings(source) {
  return {
    ...defaultSettings,
    ...source,
    general: { ...defaultSettings.general, ...(source.general || {}) },
    labels: { ...defaultSettings.labels, ...(source.labels || {}) },
    appearance: { ...defaultSettings.appearance, ...(source.appearance || {}) },
    rules: { ...defaultSettings.rules, ...(source.rules || {}) },
    platforms: { ...defaultSettings.platforms, ...(source.platforms || {}) },
    customSites: source.customSites !== undefined
      ? source.customSites
      : defaultSettings.customSites
  }
}

const I18N = {
  zh: {
    pageTitle: 'ChatTOC 设置',
    brandTitle: 'ChatTOC',
    brandSubtitle: 'AI 对话目录，一键定位',
    navGeneral: '通用',
    navAppearance: '外观与交互',
    navRules: '内容规则',
    navPlatforms: '平台与适配',
    shortcutsTitle: '快捷键',
    shortcutSearch: '聚焦搜索',
    shortcutBookmark: '收藏当前消息',
    shortcutNavigate: '上一条 / 下一条',
    shortcutClose: '关闭搜索 / 侧边栏',
    heroTitle: 'ChatTOC 设置',
    heroSubtitle: '统一 ChatGPT / Claude / Gemini / 豆包 的对话目录体验',
    heroStatWidth: '默认宽度',
    heroStatSearch: '搜索延迟',
    heroStatVirtual: '虚拟列表',
    heroStatVirtualOn: '已启用',
    generalTitle: '通用设置',
    generalStatus: '插件状态',
    generalEnableSidebar: '启用侧边栏',
    generalLanguage: '语言偏好',
    generalLanguageLabel: '界面语言',
    languageAuto: '跟随浏览器',
    languageZh: '中文',
    languageEn: 'English',
    generalLanguageHint: '主要影响提示文案与快捷说明。',
    appearanceTitle: '外观与交互',
    appearanceNameTitle: '名称设置',
    appearanceUserLabel: '用户名称',
    appearanceUserPlaceholder: '例如：用户',
    appearanceAssistantLabel: 'Assistant 名称',
    appearanceAssistantPlaceholder: '例如：Assistant',
    appearanceLayoutTitle: '布局位置',
    appearancePositionRight: '右侧（默认）',
    appearancePositionLeft: '左侧',
    appearanceWidth: '面板宽度 (px)',
    appearanceThemeTitle: '视觉风格',
    appearanceThemeAuto: '跟随网页',
    appearanceThemeLight: '浅色',
    appearanceThemeDark: '深色',
    appearanceFontSize: '字体密度',
    appearanceFontSmall: '紧凑',
    appearanceFontMedium: '标准',
    appearanceFontLarge: '宽松',
    appearanceScrollTitle: '滚动体验',
    appearanceSmoothScroll: '平滑滚动定位',
    appearanceScrollSpy: '高亮当前阅读位置',
    appearanceMessageTitle: '消息样式',
    appearanceRoleStyle: '区分用户/AI颜色',
    rulesTitle: '目录内容规则',
    rulesSummaryTitle: '摘要长度',
    rulesMaxLength: '最大字符数',
    rulesMaxLengthHint: '建议 120-160，适配两行预览。',
    rulesMinLength: '过滤短消息',
    rulesShowNumber: '显示消息序号',
    platformsTitle: '平台与适配',
    platformsPresetTitle: '预设平台',
    platformsCustomTitle: '自定义站点',
    platformsCustomHint: '用于未内置的平台：先填域名，再选模板自动生成选择器，必要时微调。',
    platformsAddSite: '+ 添加自定义站点',
    modalTitleAdd: '添加自定义站点',
    modalTitleEdit: '编辑站点',
    modalTemplateLabel: '选择模板',
    modalTemplateHint: '模板会自动填充选择器，适合结构相近的站点。',
    modalTemplateNone: '不使用模板',
    modalTemplateChatGPT: 'ChatGPT / 通用',
    modalTemplateClaude: 'Claude',
    modalTemplateGemini: 'Gemini',
    modalTemplateDeepSeek: 'DeepSeek',
    modalTemplateDoubao: '豆包',
    modalTemplateGrok: 'Grok',
    modalSiteName: '站点名称',
    modalSiteNamePlaceholder: '例如: ChatGPT',
    modalDomain: '域名',
    modalDomainPlaceholder: '例如: chat.openai.com',
    modalDomainHint: '只填写域名，不需要包含 https://',
    modalContainer: '容器选择器',
    modalContainerPlaceholder: '例如: main',
    modalItem: '消息项选择器',
    modalItemPlaceholder: "例如: [data-testid*='conversation-turn']",
    modalUserMessage: '用户消息选择器',
    modalUserMessagePlaceholder: "例如: [data-message-author-role='user']",
    modalText: '文本选择器',
    modalTextPlaceholder: "例如: [data-message-author-role='user'] .markdown",
    modalExclude: '排除选择器（可选）',
    modalExcludePlaceholder: "例如: [data-testid*='error']",
    modalEnabled: '启用此站点',
    modalCancel: '取消',
    modalSave: '保存',
    modalConfirmOverwrite: '将用模板覆盖已填写的选择器，是否继续？',
    modalValidationBasic: '请填写站点名称和域名',
    modalValidationSelectors: '请选择模板或填写必需的选择器',
    deleteConfirm: '确定要删除这个自定义站点吗？',
    customSitesEmpty: '暂无自定义站点，点击上方按钮添加',
    saved: '已保存'
  },
  en: {
    pageTitle: 'ChatTOC Settings',
    brandTitle: 'ChatTOC',
    brandSubtitle: 'AI chat TOC, find fast',
    navGeneral: 'General',
    navAppearance: 'Appearance',
    navRules: 'Content Rules',
    navPlatforms: 'Platforms',
    shortcutsTitle: 'Shortcuts',
    shortcutSearch: 'Focus search',
    shortcutBookmark: 'Bookmark message',
    shortcutNavigate: 'Previous / Next',
    shortcutClose: 'Close search / sidebar',
    heroTitle: 'ChatTOC Sidebar Experience',
    heroSubtitle: 'Unified chat TOC for ChatGPT / Claude / Gemini / Doubao',
    heroStatWidth: 'Default width',
    heroStatSearch: 'Search delay',
    heroStatVirtual: 'Virtual list',
    heroStatVirtualOn: 'Enabled',
    generalTitle: 'General',
    generalStatus: 'Plugin status',
    generalEnableSidebar: 'Enable sidebar',
    generalLanguage: 'Language',
    generalLanguageLabel: 'Interface language',
    languageAuto: 'Follow browser',
    languageZh: 'Chinese',
    languageEn: 'English',
    generalLanguageHint: 'Affects UI copy and shortcut hints.',
    appearanceTitle: 'Appearance',
    appearanceNameTitle: 'Role labels',
    appearanceUserLabel: 'User label',
    appearanceUserPlaceholder: 'e.g. User',
    appearanceAssistantLabel: 'Assistant label',
    appearanceAssistantPlaceholder: 'e.g. Assistant',
    appearanceLayoutTitle: 'Position',
    appearancePositionRight: 'Right (default)',
    appearancePositionLeft: 'Left',
    appearanceWidth: 'Panel width (px)',
    appearanceThemeTitle: 'Theme',
    appearanceThemeAuto: 'Match site',
    appearanceThemeLight: 'Light',
    appearanceThemeDark: 'Dark',
    appearanceFontSize: 'Density',
    appearanceFontSmall: 'Compact',
    appearanceFontMedium: 'Standard',
    appearanceFontLarge: 'Comfort',
    appearanceScrollTitle: 'Scrolling',
    appearanceSmoothScroll: 'Smooth scroll to message',
    appearanceScrollSpy: 'Highlight reading position',
    appearanceMessageTitle: 'Message styles',
    appearanceRoleStyle: 'Differentiate user/AI colors',
    rulesTitle: 'Message Rules',
    rulesSummaryTitle: 'Summary length',
    rulesMaxLength: 'Max characters',
    rulesMaxLengthHint: 'Suggested 120–160 for two lines.',
    rulesMinLength: 'Filter short messages',
    rulesShowNumber: 'Show message number',
    platformsTitle: 'Platforms',
    platformsPresetTitle: 'Preset platforms',
    platformsCustomTitle: 'Custom sites',
    platformsCustomHint: 'For non-built-in sites: enter domain, pick a template to auto-fill selectors, then tweak if needed.',
    platformsAddSite: '+ Add custom site',
    modalTitleAdd: 'Add custom site',
    modalTitleEdit: 'Edit site',
    modalTemplateLabel: 'Choose template',
    modalTemplateHint: 'Templates auto-fill selectors for similar layouts.',
    modalTemplateNone: 'No template',
    modalTemplateChatGPT: 'ChatGPT / Generic',
    modalTemplateClaude: 'Claude',
    modalTemplateGemini: 'Gemini',
    modalTemplateDeepSeek: 'DeepSeek',
    modalTemplateDoubao: 'Doubao',
    modalTemplateGrok: 'Grok',
    modalSiteName: 'Site name',
    modalSiteNamePlaceholder: 'e.g. ChatGPT',
    modalDomain: 'Domain',
    modalDomainPlaceholder: 'e.g. chat.openai.com',
    modalDomainHint: 'Use domain only, without https://',
    modalContainer: 'Container selector',
    modalContainerPlaceholder: 'e.g. main',
    modalItem: 'Message selector',
    modalItemPlaceholder: "e.g. [data-testid*='conversation-turn']",
    modalUserMessage: 'User message selector',
    modalUserMessagePlaceholder: "e.g. [data-message-author-role='user']",
    modalText: 'Text selector',
    modalTextPlaceholder: "e.g. [data-message-author-role='user'] .markdown",
    modalExclude: 'Exclude selector (optional)',
    modalExcludePlaceholder: "e.g. [data-testid*='error']",
    modalEnabled: 'Enable this site',
    modalCancel: 'Cancel',
    modalSave: 'Save',
    modalConfirmOverwrite: 'Replace current selectors with the template?',
    modalValidationBasic: 'Please fill site name and domain',
    modalValidationSelectors: 'Choose a template or fill required selectors',
    deleteConfirm: 'Delete this custom site?',
    customSitesEmpty: 'No custom sites yet. Click the button above to add one.',
    saved: 'Saved'
  }
}

function resolveLanguage(lang) {
  if (lang === 'auto') {
    const locale = navigator.language || 'zh'
    return locale.toLowerCase().startsWith('zh') ? 'zh' : 'en'
  }
  return lang || 'zh'
}

function getDict() {
  return I18N[resolveLanguage(settings.general.language)] || I18N.zh
}

const TEMPLATE_SELECTORS = {
  chatgpt: {
    container: 'main',
    item: "[data-testid*='conversation-turn']",
    userMessage: "[data-message-author-role='user']",
    text: "[data-message-author-role='user'] .markdown",
    exclude: "[data-testid*='error']"
  },
  claude: {
    container: 'main, [role="main"], #root, div.overflow-y-scroll.overflow-x-hidden.pt-6.flex-1',
    item: '[data-testid="user-message"], .standard-markdown',
    userMessage: '[data-testid="user-message"]',
    text: '[data-testid="user-message"], .standard-markdown',
    exclude: '[data-testid="message-warning"], [class*="Error"]'
  },
  gemini: {
    container: 'chat-window-content',
    item: '.conversation-container',
    userMessage: 'user-query',
    text: 'user-query .query-text, model-response .markdown, model-response .markdown-main-panel',
    exclude: '.error'
  },
  deepseek: {
    container: "main, [role='main'], #root, .app-container",
    item: '.ds-message',
    userMessage: '.ds-message',
    text: '.ds-message',
    exclude: "[class*='error'], [class*='Error']"
  },
  doubao: {
    container: ".inter-H_fm37, main, [role='main'], #root",
    item: "[data-testid='message_content']",
    userMessage: "[data-testid='send_message']",
    text: "[data-testid='message_text_content']",
    exclude: "[class*='error'], [class*='Error']"
  },
  grok: {
    container: 'main, [role="main"], #root, body',
    item: 'div[id^="response-"]',
    userMessage: '.items-end',
    text: '.response-content-markdown',
    exclude: ''
  }
}

function applyLanguage(lang) {
  const dict = I18N[resolveLanguage(lang)] || I18N.zh
  $(document).prop('title', dict.pageTitle)

  $('[data-i18n]').each((_, el) => {
    const key = $(el).data('i18n')
    if (dict[key]) {
      $(el).text(dict[key])
    }
  })

  $('[data-i18n-placeholder]').each((_, el) => {
    const key = $(el).data('i18nPlaceholder')
    if (dict[key]) {
      $(el).attr('placeholder', dict[key])
    }
  })
}

// ==================== 存储管理器 ====================
const StorageManager = {
  async getSettings() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['settings'], (result) => {
        resolve(normalizeSettings(result.settings || {}))
      })
    })
  },

  async saveSettings(settings) {
    return new Promise((resolve) => {
      chrome.storage.sync.set({ settings }, () => {
        resolve()
      })
    })
  },

  async updateSettings(updates) {
    const current = await this.getSettings()
    const updated = {
      general: { ...current.general, ...(updates.general || {}) },
      labels: { ...current.labels, ...(updates.labels || {}) },
      appearance: { ...current.appearance, ...(updates.appearance || {}) },
      rules: { ...current.rules, ...(updates.rules || {}) },
      platforms: { ...current.platforms, ...(updates.platforms || {}) },
      customSites: updates.customSites !== undefined
        ? updates.customSites
        : current.customSites
    }
    await this.saveSettings(updated)
  }
}

// ==================== 应用状态 ====================
let settings = { ...defaultSettings }
let activeTab = 'general'
let editingSite = null

// ==================== 显示保存提示 ====================
function showSaveIndicator() {
  const $indicator = $('#save-indicator')
  if ($indicator.length) {
    $indicator.show()
    setTimeout(() => {
      $indicator.hide()
    }, 2000)
  }
}

// ==================== 保存设置 ====================
async function saveSettings() {
  await StorageManager.saveSettings(settings)
  showSaveIndicator()
}

// ==================== 标签页切换 ====================
function switchTab(tabId) {
  activeTab = tabId

  // 更新导航状态
  $('.nav-item').each((_, item) => {
    $(item).toggleClass('active', $(item).data('tab') === tabId)
  })

  // 更新内容显示
  $('.tab-content').each((_, content) => {
    $(content).toggleClass('active', $(content).attr('id') === `tab-${tabId}`)
  })
}

// ==================== 绑定标签页切换 ====================
$('.nav-item').on('click', (e) => {
  e.preventDefault()
  switchTab($(e.currentTarget).data('tab'))
})

// ==================== 绑定设置控件 ====================
function bindSettings() {
  // 通用设置
  const $enabledCheckbox = $('#setting-enabled')
  const $languageSelect = $('#setting-language')
  const $userLabelInput = $('#setting-userLabel')
  const $assistantLabelInput = $('#setting-assistantLabel')

  if ($enabledCheckbox.length) {
    $enabledCheckbox.prop('checked', settings.general.enabled)
    $enabledCheckbox.on('change', () => {
      settings.general.enabled = $enabledCheckbox.prop('checked')
      saveSettings()
    })
  }

  if ($languageSelect.length) {
    $languageSelect.val(settings.general.language)
    $languageSelect.on('change', () => {
      settings.general.language = $languageSelect.val()
      applyLanguage(settings.general.language)
      saveSettings()
    })
  }

  if ($userLabelInput.length) {
    $userLabelInput.val(settings.labels.user)
    $userLabelInput.on('change', () => {
      settings.labels.user = $userLabelInput.val().trim() || '用户'
      saveSettings()
    })
  }

  if ($assistantLabelInput.length) {
    $assistantLabelInput.val(settings.labels.assistant)
    $assistantLabelInput.on('change', () => {
      settings.labels.assistant = $assistantLabelInput.val().trim() || 'Assistant'
      saveSettings()
    })
  }

  // 外观设置
  const $positionRadios = $('input[name="position"]')
  const $widthInput = $('#setting-width')
  const $themeRadios = $('input[name="theme"]')
  const $fontSizeSelect = $('#setting-fontSize')
  const $smoothScrollCheckbox = $('#setting-smoothScroll')
  const $scrollSpyCheckbox = $('#setting-scrollSpy')
  const $roleStyleCheckbox = $('#setting-roleStyle')

  $positionRadios.each((_, radio) => {
    const $radio = $(radio)
    if ($radio.val() === settings.appearance.position) {
      $radio.prop('checked', true)
    }
    $radio.on('change', () => {
      if ($radio.prop('checked')) {
        settings.appearance.position = $radio.val()
        saveSettings()
      }
    })
  })

  if ($widthInput.length) {
    $widthInput.val(settings.appearance.width)
    $widthInput.on('change', () => {
      settings.appearance.width = parseInt($widthInput.val(), 10)
      saveSettings()
    })
  }

  $themeRadios.each((_, radio) => {
    const $radio = $(radio)
    if ($radio.val() === settings.appearance.theme) {
      $radio.prop('checked', true)
    }
    $radio.on('change', () => {
      if ($radio.prop('checked')) {
        settings.appearance.theme = $radio.val()
        saveSettings()
      }
    })
  })

  if ($fontSizeSelect.length) {
    $fontSizeSelect.val(settings.appearance.fontSize)
    $fontSizeSelect.on('change', () => {
      settings.appearance.fontSize = $fontSizeSelect.val()
      saveSettings()
    })
  }

  if ($smoothScrollCheckbox.length) {
    $smoothScrollCheckbox.prop('checked', settings.appearance.smoothScroll)
    $smoothScrollCheckbox.on('change', () => {
      settings.appearance.smoothScroll = $smoothScrollCheckbox.prop('checked')
      saveSettings()
    })
  }

  if ($scrollSpyCheckbox.length) {
    $scrollSpyCheckbox.prop('checked', settings.appearance.scrollSpy)
    $scrollSpyCheckbox.on('change', () => {
      settings.appearance.scrollSpy = $scrollSpyCheckbox.prop('checked')
      saveSettings()
    })
  }

  if ($roleStyleCheckbox.length) {
    $roleStyleCheckbox.prop('checked', settings.appearance.roleStyle)
    $roleStyleCheckbox.on('change', () => {
      settings.appearance.roleStyle = $roleStyleCheckbox.prop('checked')
      saveSettings()
    })
  }

  // 规则设置
  const $maxLengthInput = $('#setting-maxLength')
  const $minLengthInput = $('#setting-minLength')
  const $showNumberCheckbox = $('#setting-showNumber')

  if ($maxLengthInput.length) {
    $maxLengthInput.val(settings.rules.maxLength)
    $maxLengthInput.on('change', () => {
      settings.rules.maxLength = parseInt($maxLengthInput.val(), 10)
      saveSettings()
    })
  }

  if ($minLengthInput.length) {
    $minLengthInput.val(settings.rules.minLength)
    $minLengthInput.on('change', () => {
      settings.rules.minLength = parseInt($minLengthInput.val(), 10)
      saveSettings()
    })
  }

  if ($showNumberCheckbox.length) {
    $showNumberCheckbox.prop('checked', settings.rules.showNumber)
    $showNumberCheckbox.on('change', () => {
      settings.rules.showNumber = $showNumberCheckbox.prop('checked')
      saveSettings()
    })
  }
}

// ==================== 渲染平台列表 ====================
function renderPlatforms() {
  const $container = $('#platforms-list')
  if (!$container.length) return

  $container.html(Object.keys(settings.platforms).map(domain => {
    const enabled = settings.platforms[domain]
    return `
      <div class="platform-item">
        <label class="toggle">
          <input type="checkbox" data-domain="${domain}" ${enabled ? 'checked' : ''}>
          <span class="toggle__track"></span>
          <span class="toggle__label">${domain}</span>
        </label>
      </div>
    `
  }).join(''))

  $container.find('input[type="checkbox"]').on('change', async (event) => {
    const $checkbox = $(event.currentTarget)
    const domain = $checkbox.data('domain')
    settings.platforms[domain] = $checkbox.prop('checked')
    await saveSettings()
  })
}

// ==================== 渲染自定义站点列表 ====================
function renderCustomSites() {
  const $container = $('#custom-sites-list')
  if (!$container.length) return

  if (settings.customSites.length === 0) {
    const dict = getDict()
    $container.html(`<p class="muted-text">${dict.customSitesEmpty}</p>`)
    return
  }

  $container.html(settings.customSites.map(site => {
    return `
      <div class="custom-site-item">
        <div class="site-header">
          <label class="toggle">
            <input type="checkbox" data-site-id="${site.id}" ${site.enabled ? 'checked' : ''}>
            <span class="toggle__track"></span>
            <span class="toggle__label">${site.site_name} (${site.domain})</span>
          </label>
          <div class="site-actions">
            <button class="btn-edit" data-site-id="${site.id}">编辑</button>
            <button class="btn-delete" data-site-id="${site.id}">删除</button>
          </div>
        </div>
      </div>
    `
  }).join(''))

  $container.find('input[type="checkbox"]').on('change', async (event) => {
    const $checkbox = $(event.currentTarget)
    const siteId = $checkbox.data('siteId')
    const site = settings.customSites.find(s => s.id === siteId)
    if (site) {
      site.enabled = $checkbox.prop('checked')
      await saveSettings()
    }
  })

  $container.find('.btn-edit').on('click', (event) => {
    const siteId = $(event.currentTarget).data('siteId')
    const site = settings.customSites.find(s => s.id === siteId)
    if (site) {
      editCustomSite(site)
    }
  })

  $container.find('.btn-delete').on('click', async (event) => {
    const dict = getDict()
    const siteId = $(event.currentTarget).data('siteId')
    if (confirm(dict.deleteConfirm)) {
      settings.customSites = settings.customSites.filter(s => s.id !== siteId)
      await saveSettings()
      renderCustomSites()
    }
  })
}

function applyTemplate(templateKey, { force = false } = {}) {
  const template = TEMPLATE_SELECTORS[templateKey]
  if (!template) return

  const fields = {
    container: $('#modal-container'),
    item: $('#modal-item'),
    userMessage: $('#modal-user-message'),
    text: $('#modal-text'),
    exclude: $('#modal-exclude')
  }

  const hasValue = Object.values(fields).some(field => field.length && field.val().trim())
  if (hasValue && !force) {
    const dict = getDict()
    if (!confirm(dict.modalConfirmOverwrite)) {
      return
    }
  }

  if (fields.container.length) fields.container.val(template.container || '')
  if (fields.item.length) fields.item.val(template.item || '')
  if (fields.userMessage.length) fields.userMessage.val(template.userMessage || '')
  if (fields.text.length) fields.text.val(template.text || '')
  if (fields.exclude.length) fields.exclude.val(template.exclude || '')
}

// ==================== 打开模态框 ====================
function openModal(site = null) {
  editingSite = site
  const $modal = $('#modal-overlay')
  const $title = $('#modal-title')
  const dict = getDict()

  if ($modal.length && $title.length) {
    $title.text(site ? dict.modalTitleEdit : dict.modalTitleAdd)

    if (site) {
      $('#modal-site-name').val(site.site_name || '')
      $('#modal-domain').val(site.domain || '')
      $('#modal-container').val(site.selectors?.container || '')
      $('#modal-item').val(site.selectors?.item || '')
      $('#modal-user-message').val(site.selectors?.userMessage || '')
      $('#modal-text').val(site.selectors?.text || '')
      $('#modal-exclude').val(site.selectors?.exclude || '')
      $('#modal-enabled').prop('checked', site.enabled || false)
    } else {
      $('#modal-site-name').val('')
      $('#modal-domain').val('')
      $('#modal-container').val('')
      $('#modal-item').val('')
      $('#modal-user-message').val('')
      $('#modal-text').val('')
      $('#modal-exclude').val('')
      $('#modal-enabled').prop('checked', true)
    }

    $('#modal-template').val('')

    $modal.css('display', 'flex')
  }
}

// ==================== 关闭模态框 ====================
function closeModal() {
  const $modal = $('#modal-overlay')
  if ($modal.length) {
    $modal.hide()
  }
  editingSite = null
}

// ==================== 保存自定义站点 ====================
async function saveCustomSite() {
  const dict = getDict()
  const siteName = $('#modal-site-name').val().trim()
  const domain = $('#modal-domain').val().trim()
  const templateKey = $('#modal-template').val() || ''
  let container = $('#modal-container').val().trim()
  let item = $('#modal-item').val().trim()
  let userMessage = $('#modal-user-message').val().trim()
  let text = $('#modal-text').val().trim()
  let exclude = $('#modal-exclude').val().trim()
  const enabled = $('#modal-enabled').prop('checked')

  if (!siteName || !domain) {
    alert(dict.modalValidationBasic)
    return
  }

  if ((!container || !item || !userMessage || !text) && TEMPLATE_SELECTORS[templateKey]) {
    applyTemplate(templateKey, { force: true })
    container = $('#modal-container').val().trim()
    item = $('#modal-item').val().trim()
    userMessage = $('#modal-user-message').val().trim()
    text = $('#modal-text').val().trim()
    exclude = $('#modal-exclude').val().trim()
  }

  if (!container || !item || !userMessage || !text) {
    alert(dict.modalValidationSelectors)
    return
  }

  const siteData = {
    site_name: siteName,
    domain: domain,
    selectors: {
      container,
      item,
      userMessage,
      text,
      exclude: exclude || undefined
    },
    enabled
  }

  if (editingSite) {
    const index = settings.customSites.findIndex(s => s.id === editingSite.id)
    if (index !== -1) {
      settings.customSites[index] = {
        ...editingSite,
        ...siteData
      }
    }
  } else {
    settings.customSites.push({
      id: `custom-${Date.now()}`,
      ...siteData
    })
  }

  await saveSettings()
  renderCustomSites()
  closeModal()
}

// ==================== 编辑自定义站点 ====================
function editCustomSite(site) {
  openModal(site)
}

// ==================== 绑定模态框事件 ====================
$('#btn-add-site').on('click', () => {
  openModal()
})

$('#modal-close').on('click', closeModal)
$('#modal-cancel').on('click', closeModal)
$('#modal-save').on('click', saveCustomSite)
$('#modal-template').on('change', (event) => {
  const templateKey = $(event.currentTarget).val()
  if (templateKey) {
    applyTemplate(templateKey)
  }
})

$('#modal-overlay').on('click', (event) => {
  if ($(event.target).attr('id') === 'modal-overlay') {
    closeModal()
  }
})

// ==================== 初始化 ====================
async function init() {
  settings = await StorageManager.getSettings()
  applyLanguage(settings.general.language)
  bindSettings()
  renderPlatforms()
  renderCustomSites()
}

// 页面加载完成后初始化
$(init)
